---
title: Tunnel permissions · Cloudflare Zero Trust docs
description: A remotely-managed tunnel only requires the tunnel token to run.
  Anyone with access to the token will be able to run the tunnel.
lastUpdated: 2025-10-27T16:36:33.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/cloudflare-one/networks/connectors/cloudflare-tunnel/configure-tunnels/remote-tunnel-permissions/
  md: https://developers.cloudflare.com/cloudflare-one/networks/connectors/cloudflare-tunnel/configure-tunnels/remote-tunnel-permissions/index.md
---

A remotely-managed tunnel only requires the tunnel token to run. Anyone with access to the token will be able to run the tunnel.

## Get the tunnel token

To get the token for a remotely-managed tunnel:

* Dashboard

  1. In [Cloudflare One](https://one.dash.cloudflare.com/), go to **Networks** > **Connectors** > **Cloudflare Tunnels**.
  2. Select a `cloudflared` tunnel and select **Edit**.
  3. Copy the `cloudflared` installation command.
  4. Paste the installation command into any text editor. The token value is of the form `eyJhIjoiNWFiNGU5Z...`

* API

  Make a `GET` request to the [Cloudflare Tunnel token](https://developers.cloudflare.com/api/resources/zero_trust/subresources/tunnels/subresources/cloudflared/subresources/token/methods/get/) endpoint:

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Cloudflare One Connectors Write`
  * `Cloudflare One Connector: cloudflared Write`
  * `Cloudflare Tunnel Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/cfd_tunnel/$TUNNEL_ID/token" \
    --request GET \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN"
  ```

  ```sh
  {
    "success": true,
    "errors": [],
    "messages": [],
    "result": "eyJhIjoiNWFiNGU5Z..."
  }
  ```

  The token value can be found in the `result`.

* Terraform (v5)

  ```tf
  data "cloudflare_zero_trust_tunnel_cloudflared_token" "tunnel_token" {
    account_id = var.cloudflare_account_id
    tunnel_id = cloudflare_zero_trust_tunnel_cloudflared.example_tunnel.id
  }
  ```

  If your host machine is not managed in Terraform or you want to install the tunnel manually, you can output the token value to the CLI.

  Example: Output to CLI

  1. Output the tunnel token to the Terraform state file:

     ```tf
     output "tunnel_token" {
       value       = data.cloudflare_zero_trust_tunnel_cloudflared_token.tunnel_token.token
       sensitive   = true
     }
     ```

  2. Apply the configuration:

     ```sh
     terraform apply
     ```

  3. Read the tunnel token:

     ```sh
     terraform output -raw tunnel_token
     ```

     ```sh
     eyJhIj...
     ```

  Alternatively, pass `data.cloudflare_zero_trust_tunnel_cloudflared_token.tunnel_token.token` directly into your host's Terraform configuration or store the token in your secret management tool.

  Example: Store in HashiCorp Vault

  ```tf
  resource "vault_generic_secret" "tunnel_token" {
    path         = "kv/cloudflare/tunnel_token"


    data_json = jsonencode({
      "TUNNEL_TOKEN" = data.cloudflare_zero_trust_tunnel_cloudflared_token.tunnel_token.token
    })
  }
  ```

## Rotate a token without service disruption

Cloudflare recommends rotating the tunnel token at a regular cadence to reduce the risk of token compromise. You can rotate a token with minimal disruption to users as long as the tunnel is served by at least two [`cloudflared` replicas](https://developers.cloudflare.com/cloudflare-one/networks/connectors/cloudflare-tunnel/configure-tunnels/tunnel-availability/). To ensure service availability, we recommend performing token rotations outside of working hours or in a maintenance window.

To rotate a tunnel token:

1. Refresh the token on Cloudflare:

* Dashboard

  1. In [Cloudflare One](https://one.dash.cloudflare.com/), go to **Networks** > **Connectors** > **Cloudflare Tunnels**.
  2. Select a `cloudflared` tunnel and select **Edit**.
  3. Select **Refresh token**.
  4. Copy the `cloudflared` installation command for your operating system. This command contains the new token.

* API

  1. Generate a random base64 string (minimum size 32 bytes) to use as a tunnel secret:

     ```sh
     openssl rand -base64 32
     ```

     ```sh
     AQIDBAUGBwgBAgMEBQYHCAECAwQFBgcIAQIDBAUGBwg=
     ```

  2. Make a `PATCH` request to the [Cloudflare Tunnel](https://developers.cloudflare.com/api/resources/zero_trust/subresources/tunnels/methods/edit/) endpoint:

     Required API token permissions

     At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

     * `Cloudflare One Connectors Write`
     * `Cloudflare One Connector: cloudflared Write`
     * `Cloudflare Tunnel Write`

     ```bash
     curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/cfd_tunnel/$TUNNEL_ID" \
       --request PATCH \
       --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
       --json '{
         "name": "Example tunnel",
         "tunnel_secret": "AQIDBAUGBwgBAgMEBQYHCAECAwQFBgcIAQIDBAUGBwg="
       }'
     ```

     ```sh
     {
       "success": true,
       "errors": [],
       "messages": [],
       "result": {
         "id": "f70ff985-a4ef-4643-bbbc-4a0ed4fc8415",
         "account_tag": "699d98642c564d2e855e9661899b7252",
         "created_at": "2024-12-04T22:03:26.291225Z",
         "deleted_at": null,
         "name": "Example tunnel",
         "connections": [],
         "conns_active_at": null,
         "conns_inactive_at": "2024-12-04T22:03:26.291225Z",
         "tun_type": "cfd_tunnel",
         "metadata": {},
         "status": "inactive",
         "remote_config": true,
         "token": "eyJhIjoiNWFiNGU5Z..."
       }
     }
     ```

  3. Copy the `token` value shown in the output.

After refreshing the token, `cloudflared` can no longer establish new connections to Cloudflare using the old token. However, existing connectors will remain active and the tunnel will continue serving traffic.

1. On half of your `cloudflared` replicas, update `cloudflared` to use the new token. For example, on a Linux host:

   ```sh
   sudo cloudflared service install <TOKEN>
   ```

2. Restart `cloudflared`:

   ```sh
   sudo systemctl restart cloudflared.service
   ```

3. Confirm that the service started correctly:

   ```sh
   sudo systemctl status cloudflared
   ```

   While these replicas are connecting to Cloudflare with the new token, traffic will automatically route through the other replicas.

4. Wait 10 minutes for traffic to route through the new connectors.

5. Repeat steps 2, 3, and 4 for the second half of the replicas.

The tunnel token is now fully rotated. The old token is no longer in use.

## Rotate a compromised token

If your tunnel token is compromised, we recommend taking the following steps:

1. Refresh the token using the dashboard or API. Refer to Step 1 of [Rotate a token without service disruption](#rotate-a-token-without-service-disruption).

2. [Delete all connections](https://developers.cloudflare.com/api/resources/zero_trust/subresources/tunnels/subresources/connections/methods/delete/) between `cloudflared` and Cloudflare:

   Required API token permissions

   At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

   * `Cloudflare One Connectors Write`
   * `Cloudflare One Connector: cloudflared Write`
   * `Cloudflare Tunnel Write`

   ```bash
   curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/cfd_tunnel/$TUNNEL_ID/connections" \
     --request DELETE \
     --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN"
   ```

   This will clean up any unauthorized connections and prevent users from connecting to your network.

3. On each `cloudflared` replica, update `cloudflared` to use the new token. For example, on a Linux host:

   ```sh
   sudo cloudflared service install <TOKEN>
   ```

4. Restart `cloudflared`:

   ```sh
   sudo systemctl restart cloudflared.service
   ```

5. Confirm that the service started correctly:

   ```sh
   sudo systemctl status cloudflared
   ```

The tunnel token is now fully rotated. The old token is no longer in use.

## Account-scoped roles

Minimum permissions needed to create, delete, and configure tunnels for an account:

* [Cloudflare Access](https://developers.cloudflare.com/cloudflare-one/roles-permissions/)

Additional permissions needed to [route traffic to a public hostname](https://developers.cloudflare.com/cloudflare-one/networks/connectors/cloudflare-tunnel/routing-to-tunnel/):

* [DNS](https://developers.cloudflare.com/fundamentals/manage-members/roles/)
